﻿using System.Collections.Generic;
using UnityEngine;

public class OctreeTreeNode
{
    /// <summary>
    /// 分割单元最小尺寸
    /// </summary>
    private float minSize;

    /// <summary>
    /// 最大递归深度
    /// </summary>
    private int maxDepth;

    /// <summary>
    /// 当前节点深度
    /// </summary>
    private int depth;

    /// <summary>
    /// 父节点
    /// </summary>
    public OctreeTreeNode parent { get; private set; }

    /// <summary>
    /// 子节点列表
    /// </summary>
    public List<OctreeTreeNode> child;

    /// <summary>
    /// 包围盒作为节点方块
    /// </summary>
    public Bounds nodeCube { get; private set; }

    /// <summary>
    /// 是否包含障碍物（与碰撞体相交）
    /// </summary>
    public bool isBlocked = false;

    /// <summary>
    /// 所有共享面邻居节点列表
    /// </summary>
    public List<OctreeTreeNode> neighbors = new List<OctreeTreeNode>();

    /// <summary>
    /// 获取包围盒的最小点
    /// </summary>
    public Vector3 Min => nodeCube.min;

    /// <summary>
    /// 获取包围盒的最大点
    /// </summary>
    public Vector3 Max => nodeCube.max;

    /// <summary>
    /// 判断是否为叶子节点
    /// </summary>
    public bool IsLeaf => child == null || child.Count == 0;

    /// <summary>
    /// 是否可通行（叶子节点且不包含障碍物）
    /// </summary>
    public bool IsWalkable => IsLeaf && !isBlocked;

    /// <summary>
    /// 
    /// </summary>
    /// <param name="parent">父节点</param>
    /// <param name="nodeCube">包围盒</param>
    /// <param name="depth">当前递归深度</param>
    /// <param name="minSize">最小分割尺寸</param>
    /// <param name="maxDepth">最大递归深度</param>
    public OctreeTreeNode(OctreeTreeNode parent, Bounds nodeCube, int depth, float minSize, int maxDepth)
    {
        this.parent = parent;
        this.nodeCube = nodeCube;
        this.depth = depth;
        this.minSize = minSize;
        this.maxDepth = maxDepth;
    }

    public void Divide(List<Collider> colliders)
    {
        if (nodeCube.size.x / 2 < minSize || depth >= maxDepth)
        {
            // 叶子节点：检查是否与障碍物相交
            isBlocked = colliders.Exists(c => nodeCube.Intersects(c.bounds));
            return;
        }

        // 找出与当前节点包围盒相交的碰撞体
        var intersecting = colliders.FindAll(c => nodeCube.Intersects(c.bounds));
        // 前剪枝：没有物体相交则停止
        if (intersecting.Count == 0)
        {
            return;
        }

        float childHalfSize = nodeCube.size.x / 4;
        if (child == null)
        {
            child = new List<OctreeTreeNode>();
        }
        Vector3 offest; // 子节点相对于父节点的偏移量
        for (int i = 0; i < 8; i++)
        {
            offest = new Vector3(
                ((i & 1) == 0) ? -childHalfSize : childHalfSize, // 001
                ((i & 2) == 0) ? -childHalfSize : childHalfSize, // 010
                ((i & 4) == 0) ? -childHalfSize : childHalfSize  // 100
            );
            Bounds childBounds = new Bounds(
                nodeCube.center + offest,
                new Vector3(childHalfSize * 2, childHalfSize * 2, childHalfSize * 2)
            );

            OctreeTreeNode childNode = new OctreeTreeNode(this, childBounds, depth + 1, minSize, maxDepth);
            child.Add(childNode);
            childNode.Divide(intersecting); // 只传递相交的碰撞体
        }
    }

    //isSeeOne为true，则只查看分裂后的一个，否则查看所有分裂后的方块
    public void Draw(bool isSeeOne, bool drawConnections = false, bool drawBounds = true)
    {
        // 根据是否包含障碍物设置颜色
        if (drawBounds)
        {
            if (IsLeaf)
            {
                Gizmos.color = isBlocked ? Color.red : Color.green;
            }
            else
            {
                Gizmos.color = Color.white;
            }
            Gizmos.DrawWireCube(nodeCube.center, nodeCube.size);
        }

        // 绘制邻居连接线（只绘制可通行节点的连接）
        if (drawConnections && IsWalkable)
        {
            Gizmos.color = Color.yellow;
            foreach (var neighbor in neighbors)
            {
                // 只绘制一半的连接线，避免重复绘制
                if (neighbor != null && neighbor.GetHashCode() > this.GetHashCode())
                {
                    Gizmos.DrawLine(nodeCube.center, neighbor.nodeCube.center);
                }
            }
        }

        if (child == null)
            return;
        foreach (var c in child)
        {
            c.Draw(isSeeOne, drawConnections, drawBounds);
            if (isSeeOne)
            {
                break;
            }
        }
    }
}

public class MyOctreetree
{
    public OctreeTreeNode rootNode { get; private set; }

    /// <summary>
    /// 叶子节点列表
    /// </summary>
    public List<OctreeTreeNode> leafNodes { get; private set; }

    /// <summary>
    /// 共享面最小面积阈值（小于此值不建立连接）
    /// </summary>
    private float minSharedAreaThreshold;

    /// <summary>
    /// 浮点数比较容差
    /// </summary>
    private const float EPSILON = 0.001f;

    public MyOctreetree(List<GameObject> objects, float minSize, int maxDepth, float minSharedArea = 0.01f)
    {
        if (objects == null || objects.Count == 0)
            return;

        // 用第一个物体的包围盒初始化
        var baseCube = objects[0].GetComponent<Renderer>().bounds;
        // 将剩余对象的包围盒合并
        for (int i = 1; i < objects.Count; i++)
        {
            baseCube.Encapsulate(objects[i].GetComponent<Renderer>().bounds);
        }
        // 将包围盒调整为立方体
        var cubeSize = Mathf.Max(baseCube.size.x, Mathf.Max(baseCube.size.y, baseCube.size.z)) * Vector3.one;
        baseCube.SetMinMax(baseCube.center - cubeSize / 2f, baseCube.center + cubeSize / 2f);

        rootNode = new OctreeTreeNode(null, baseCube, 0, minSize, maxDepth);

        var colliders = new List<Collider>();
        foreach (var o in objects)
        {
            if (o.TryGetComponent<Collider>(out var collider))
            {
                colliders.Add(collider);
            }
        }
        rootNode.Divide(colliders);

        // 收集叶子节点
        leafNodes = new List<OctreeTreeNode>();
        CollectLeafNodes(rootNode);

        // 设置最小共享面积阈值
        minSharedAreaThreshold = minSharedArea;

        // 基于共享面为可通行节点建立邻居连接
        BuildNeighborConnectionsBySharedFace();
    }

    /// <summary>
    /// 递归收集所有叶子节点
    /// </summary>
    private void CollectLeafNodes(OctreeTreeNode node)
    {
        if (node == null) return;

        // 只收集不含障碍物的叶子节点
        if (node.IsLeaf)
        {
            if (node.isBlocked)
            {
                return;
            }
            leafNodes.Add(node);
            return;
        }

        // 非叶子节点，递归处理子节点
        foreach (var c in node.child)
        {
            CollectLeafNodes(c);
        }
    }

    /// <summary>
    /// 基于共享面为叶子节点建立邻居连接
    /// 使用双重循环遍历所有叶子节点对，检测共享面
    /// </summary>
    private void BuildNeighborConnectionsBySharedFace()
    {
        int count = leafNodes.Count;
        for (int i = 0; i < count; i++)
        {
            for (int j = i + 1; j < count; j++)
            {
                var nodeA = leafNodes[i];
                var nodeB = leafNodes[j];

                // 检测是否有共享面
                if (HasSharedFace(nodeA, nodeB, out float sharedArea))
                {
                    // 面积大于阈值才建立连接
                    if (sharedArea >= minSharedAreaThreshold)
                    {
                        // 双向连接
                        nodeA.neighbors.Add(nodeB);
                        nodeB.neighbors.Add(nodeA);
                    }
                }
            }
        }
    }

    /// <summary>
    /// 检测两个节点是否有共享面
    /// </summary>
    /// <param name="a">节点A</param>
    /// <param name="b">节点B</param>
    /// <param name="sharedArea">输出共享面积</param>
    /// <returns>是否有共享面</returns>
    private bool HasSharedFace(OctreeTreeNode a, OctreeTreeNode b, out float sharedArea)
    {
        sharedArea = 0f;

        // 检查X轴方向的共享面 (A.max.x == B.min.x 或 A.min.x == B.max.x)
        if (Mathf.Abs(a.Max.x - b.Min.x) < EPSILON || Mathf.Abs(a.Min.x - b.Max.x) < EPSILON)
        {
            // 计算Y和Z的重叠区间
            float overlapY = CalculateOverlap(a.Min.y, a.Max.y, b.Min.y, b.Max.y);
            float overlapZ = CalculateOverlap(a.Min.z, a.Max.z, b.Min.z, b.Max.z);

            if (overlapY > 0 && overlapZ > 0)
            {
                sharedArea = overlapY * overlapZ;
                return true;
            }
        }

        // 检查Y轴方向的共享面 (A.max.y == B.min.y 或 A.min.y == B.max.y)
        if (Mathf.Abs(a.Max.y - b.Min.y) < EPSILON || Mathf.Abs(a.Min.y - b.Max.y) < EPSILON)
        {
            // 计算X和Z的重叠区间
            float overlapX = CalculateOverlap(a.Min.x, a.Max.x, b.Min.x, b.Max.x);
            float overlapZ = CalculateOverlap(a.Min.z, a.Max.z, b.Min.z, b.Max.z);

            if (overlapX > 0 && overlapZ > 0)
            {
                sharedArea = overlapX * overlapZ;
                return true;
            }
        }

        // 检查Z轴方向的共享面 (A.max.z == B.min.z 或 A.min.z == B.max.z)
        if (Mathf.Abs(a.Max.z - b.Min.z) < EPSILON || Mathf.Abs(a.Min.z - b.Max.z) < EPSILON)
        {
            // 计算X和Y的重叠区间
            float overlapX = CalculateOverlap(a.Min.x, a.Max.x, b.Min.x, b.Max.x);
            float overlapY = CalculateOverlap(a.Min.y, a.Max.y, b.Min.y, b.Max.y);

            if (overlapX > 0 && overlapY > 0)
            {
                sharedArea = overlapX * overlapY;
                return true;
            }
        }

        return false;
    }

    /// <summary>
    /// 计算两个一维区间的重叠长度
    /// </summary>
    /// <param name="aMin">区间A最小值</param>
    /// <param name="aMax">区间A最大值</param>
    /// <param name="bMin">区间B最小值</param>
    /// <param name="bMax">区间B最大值</param>
    /// <returns>重叠长度，无重叠返回0</returns>
    private float CalculateOverlap(float aMin, float aMax, float bMin, float bMax)
    {
        float overlapMin = Mathf.Max(aMin, bMin);
        float overlapMax = Mathf.Min(aMax, bMax);
        return Mathf.Max(0, overlapMax - overlapMin);
    }
}

public class CreatOctreeTree : MonoBehaviour
{
    /// <summary>
    /// 只看分裂后的一个
    /// </summary>
    public bool isSeeOne = false;

    /// <summary>
    /// 是否绘制邻居连接线
    /// </summary>
    public bool drawConnections = false;

    /// <summary>
    /// 是否绘制包围盒
    /// </summary>
    public bool drawBounds = true;

    /// <summary>
    /// 分割单元最小尺寸
    /// </summary>
    [Range(0.1f, 10f)]
    public float minSize = 1.0f;

    /// <summary>
    /// 最大递归深度
    /// </summary>
    [Range(1, 10)]
    public int maxDepth = 5;

    /// <summary>
    /// 共享面最小面积阈值（小于此值不建立连接）
    /// </summary>
    [Range(0.001f, 1f)]
    public float minSharedArea = 0.01f;

    /// <summary>
    ///  需要划分八叉树的物体列表
    /// </summary>
    public List<GameObject> objects;

    /// <summary>
    /// 八叉树
    /// </summary>
    private MyOctreetree myOctreetree;

    /// <summary>
    /// 八叉树构建时间(毫秒)
    /// </summary>
    [Header("构建信息")]
    public float buildTimeMs;

    /// <summary>
    /// 叶子节点数量
    /// </summary>
    public int leafNodeCount;

    /// <summary>
    /// 可通行节点数量
    /// </summary>
    public int walkableNodeCount;

    /// <summary>
    /// 邻居连接数量
    /// </summary>
    public int connectionCount;

    private void Awake()
    {
        RebuildOctree();
    }

    private void OnDrawGizmos()
    {
        if (Application.isPlaying && myOctreetree?.rootNode != null)
        {
            myOctreetree.rootNode.Draw(isSeeOne, drawConnections, drawBounds);
        }
    }

    /// <summary>
    /// 清除八叉树
    /// </summary>
    public void ClearOctree()
    {
        myOctreetree = null;
        objects?.Clear();
        buildTimeMs = 0;
        leafNodeCount = 0;
        walkableNodeCount = 0;
        connectionCount = 0;
        Debug.Log("八叉树已清除");
    }

    /// <summary>
    /// 重新构建八叉树
    /// </summary>
    public void RebuildOctree()
    {
        // 获取场景中有Render的GameObject
        var renderers = FindObjectsOfType<Renderer>();
        objects = new List<GameObject>();
        foreach (var renderer in renderers)
        {
            objects.Add(renderer.gameObject);
        }

        var stopwatch = System.Diagnostics.Stopwatch.StartNew();
        myOctreetree = new MyOctreetree(objects, minSize, maxDepth, minSharedArea);
        stopwatch.Stop();
        buildTimeMs = stopwatch.ElapsedMilliseconds;
        leafNodeCount = myOctreetree.leafNodes?.Count ?? 0;
        walkableNodeCount = myOctreetree.leafNodes?.Count ?? 0;

        // 统计连接数量（每条连接存储在两个节点中，所以除以2）
        connectionCount = 0;
        if (myOctreetree.leafNodes != null)
        {
            foreach (var node in myOctreetree.leafNodes)
            {
                connectionCount += node.neighbors.Count;
            }
            connectionCount /= 2;
        }

        Debug.Log($"八叉树构建完成，耗时: {buildTimeMs} ms，叶子节点: {leafNodeCount}，可通行节点: {walkableNodeCount}，连接数: {connectionCount}");
    }
}